(* Samuele Giraudo
 * 2024-10, 2024-11
 *)

type polyominos = Polyomino of (Squares.squares list)

let make sqs =
    Polyomino (sqs |> List.sort_uniq compare)

let origin =
    make [Squares.origin]

let squares p =
    let Polyomino sqs = p in
    sqs

let is_filled_square sq p =
    p |> squares |> List.mem sq

let area p =
    p |> squares |> List.length

let add_square sq p =
    make (sq :: squares p)

let extract_value sq_to_coordinate operation p =
    match p |> squares |> List.map sq_to_coordinate with
        |[] -> None
        |c :: cs -> Some (cs |> List.fold_left operation c)

let x_min p =
    extract_value Squares.x_coordinate min p

let x_max p =
    extract_value Squares.x_coordinate max p

let y_min p =
    extract_value Squares.y_coordinate min p

let y_max p =
    extract_value Squares.y_coordinate max p

let bounding_box p =
    match x_min p, x_max p, y_min p, y_max p with
        |Some x_min, Some x_max, Some y_min, Some y_max ->
            Some (Squares.make x_min y_min, Squares.make x_max y_max)
        |_ -> None

let to_string p =
    let square_to_string sq =
        if  not (is_filled_square sq p) then
            "·"
        else if sq = Squares.origin then
            "X"
        else
            "O"
    in
    match bounding_box p with
        |None -> ""
        |Some (sq_min, sq_max) ->
            let x_min = Squares.x_coordinate sq_min
            and y_min = Squares.y_coordinate sq_min
            and x_max = Squares.x_coordinate sq_max
            and y_max = Squares.y_coordinate sq_max in
            List.init (y_max - y_min + 1) (fun i -> i + y_min)
            |> List.rev
            |> List.map
                (fun y ->
                    List.init (x_max - x_min + 1) (fun i -> i + x_min)
                    |> List.map (fun x -> square_to_string (Squares.make x y))
                    |> String.concat "")
            |> String.concat "\n"

let to_prolog p =
    let picture_str =
        "/*\n" ^ to_string p ^ "\n*/\n"
    in
    let predicate_str =
        p
        |> squares
        |> List.map
            (fun sq ->
                Printf.sprintf
                    "filled(square(%d, %d))."
                    (Squares.x_coordinate sq)
                    (Squares.y_coordinate sq))
        |> String.concat "\n"
    in
    picture_str ^ "\n" ^ predicate_str

