##################################################################

######
#     #  ######   ####      #     ####   #    #  ######  #####
#     #  #       #          #    #    #  ##   #  #       #    #
#     #  #####    ####      #    #       # #  #  #####   #    #
#     #  #            #     #    #  ###  #  # #  #       #####
#     #  #       #    #     #    #    #  #   ##  #       #   #
######   ######   ####      #     ####   #    #  ######  #    #


##################################################################
#
#	GENERATION OF A GOOD 2-DESIGN
#
# By  	Vladimir Grebinski 	(grebinsk@loria.fr)
# and	Gregory Kucherov 	(kucherov@loria.fr)
#
# First appeared Jan 1996 
#
###################################################################

# 1. Introduction

# Whenever we cover all possible $k$ subsets of a set $1..n$ by some
# $l$ (>k) subsets, such that all $k$ subset are covered, one says we
# have designed (n,k,l)-coverage (=(n,k,l)-design). The lower bound on
# the minimum number of covering sets is
# $binomial(n,k)\binomial(l,k)$.  Whether it can be achieved
# asymptotically when $k$ and $l$ is fixed, is the Erdoes-Hanani
# conjecture proved by Roedl in 1985.

# While the latter proof is highly non-constructive, the need of a
# "good" practical solution is arisen. We present the realisation of
# the algorithm described in our paper (seebelow), which builds a
# "good" 2-covering of a set $1..n$. The present program claims to
# find "good" solution in the following sense:

# 1. Whenever n=k^a=p^(ab), with prime $p$, it outputs the precise
# coverage(=design).  
# 2. Otherwise it extends $n$ to the nearest
# number such that $n/k$ is a prime power. And apply itself as in
# "ideal" case.  It can be proven that such approach gives
# asymptotically optimal 2-coverage (=design).

# For more information see: 

# V.Grebinski, G. Kucherov, "Reconstructing a Hamiltonian Circuit by
# Querying the Graph: Application to DNA Physical Mapping" , Technical
# Report 96-R-123, Centre de Recherche en Informatique de Nancy
# (CRIN).

# The paper is available at
# http://www.loria.fr/~grebinsk/publications/ri96R123.ps.gz

#########################################################################
#
#2. Short description
#
# The function Design(n,k) returns a coverage of all 2-subset of the
# set [1..n] by subset of size at most k. The answer is given in form
# of list of subset, where each subset is represented as a list of its
# members.  
#
# The function DesignRND(n,k) returns a random coverage, which is
# formed by a random permutation of the entries [1..n].
#
###########################################################################

#3. Drawbacks and limitations 
# 
# The design with VERY small (n,k) is not very good. When n>>k^2,
# everything goes to its asymptotic value.


# Use Galois fields...
readlib(GF):



#################
#
#best(n) -> [p,k] -- closest to the prime power 
#
best:=proc(n)
	options remember;
	m:=2*n;
	for i in [ seq([nextprime(ceil(evalf(-1.0+n^(1/j)))),j] ,
           		j=1..ceil(evalf(1+log[2](n))))]
	do
	 	if i[1]^i[2]<m then x:=[i[1],i[2]]: m:=i[1]^i[2] fi:
	od:
	x
end:



# Design(n,k) generates a  (2,k) design of a set 1..n
Design:=proc (n,k) 
	select((L-> evalb( nops(L)>1)), Questions1(n,k)(seq(i,i=1..n)))
end:

# DesignRND(n,k) generates a random (2,k) design of a set 1..n
with(combinat):
DesignRND:=proc (n,k) 
	select((L-> evalb( nops(L)>1)), Questions1(n,k)(op(randperm(n))))
end:


##### 
# shift -- needed by Questions1
shift:=proc(t,k,G) 
	option remember;	
	[seq(G[`*`](G[input](i),G[input](t)),i=0..k-1)]:
end:


#####
# Mat -- converts element at (x,y) in a table with width $xmax$ and
# $n$ elements into $a.i$, for some $i$, or an empty element. Needed
# by Questions1
Mat:=proc(x,y,xmax,n)
 if (x>=xmax) or (y-1)*xmax+x+1>n then op([]) else a.((y-1)*xmax+x+1) fi:
end:



#################################################
#Questions1(n,k) -- the main function. 
#Returns function [list of $n$ names]-> (2,k) design on this names.

Questions1:=proc(n,k) option remember;

 V:=[seq(a.i,i=1..n)]:
 F:=floor(n/k); F1:=ceil(n/k): Filled:=n mod k:
 TB:=best(max(ceil(n/k),k));
 G:=GF(op(TB));
 T:=TB[1]^TB[2];
 N:=n;

# Case 1 (n<=k) 
# In this case all 2-subset can be covered by one $k$-set.

if 	n<=k 	then V
elif 	n<=k^2 	then

# Case 2 k^2>n => treat the set as affine space with near to $n$
# elements and traverse all its lines.

#? This is the case 3 in "this" terminology.

	xmax:=ceil(sqrt(n)): #Calculating good width
	TB:=best(xmax);
	G:=GF(op(TB)):
	T:=TB[1]^TB[2]; N:=n;
	k1:=ceil(n/xmax);	#Calculating new k
	seq1:=[seq(i, i=1..k1)]:
	f:=proc(L,xmax,k,N) [seq( Mat(L[i],i,xmax,N), i=1..k)] end: 
	M1:=
	seq(
	(	seq( f( 
		       map( (x,col,G)-> G[output](G[`+`](x,(col))),
		  	    shift(t,k1,G),
			    G[input](col),G),xmax,k1,N),
	    col=0..T-1)) , t=0..T-1),
#now recursive apply
	seq( proc (a,b) op(op((Questions1(nops(a),b)(op(a))))) end
	     (simp([seq(Mat(x,i,xmax,n),x=0..T-1)]),k),i=1..k1 ):
else
# The last case, Case 3 -- k<n<k^2. Some heuristique are implemented.

	 xmax:=ceil(n/k):
	 TB:=best(xmax);
	 G:=GF(op(TB));
	 T:=TB[1]^TB[2];
	 k1:=k;		
	 seq1:=[seq(i, i=1..k)]:
	 f:=proc(L,xmax,k,N) [seq( Mat(L[i],i,xmax,N), i=1..k)] end: 
	 seq(
		(seq( f( map( (x,col,G)-> G[output](G[`+`](x,(col))),
		  	    shift(t,k,G),
			    G[input](col),G),
		         xmax,k,N)
		   ,col=0..T-1)) , t=0..T-1),
	 seq( proc (a,b) (op((Questions1(nops(a),b)(op(a))))) end
	     (([seq(Mat(x,i,xmax,n),x=0..xmax-1)]),k),i=1..k ):
fi:
unapply(["],op(V)):
end:
######################################
#	This is the end of the package
###################################### 

####
# Example
# >Design(9,3);

# [[1, 4, 7], [2, 5, 8], [3, 6, 9], [1, 5, 9], [2, 6, 7], [3, 4, 8],
# [1, 6, 8], [ 2, 4, 9], [3, 5, 7], [1, 2, 3], [4, 5, 6], [7, 8, 9]]

# >DesignRND(12,4);

# [[10, 5, 3], [6, 8, 2], [1, 12, 7], [9, 11, 4], [10, 8, 7], [6, 5,
# 4], [1, 11, 3], [9, 12, 2], [10, 12, 4], [6, 11, 7], [1, 5, 2], [9,
# 8, 3], [10, 11, 2], [6, 12, 3], [1, 8, 4], [9, 5, 7], [10, 6, 1, 9],
# [5, 8, 12, 11], [3, 2, 7, 4]]



